/*
**
** Enumerating Hurdles
** -------------------
**
** Generating input files.
**
** Author: Dimitris Diochnos
**
** Date Started : Monday, 21st June 2004.
** Date Finished: Friday, 29th July 2004.
**
** (For better visual representation of the code use Tab Width = 4)
**
*/



#include <stdio.h>
#include <string.h>
#include <math.h>
#include <malloc.h>

/*********************************************************/

	/***********\
	* CONSTANTS *
	\***********/

/*
** Inverse Pemutation
** Number of Hurdles: 1
*/
#define SMALL_INVERSE 2500							/* Total number of elements. */
#define LARGE_INVERSE 50000							/* Total number of elements. */

/*
** Whole Sequence
** Number of Hurdles: 1
*/
#define SMALL_WHOLE 2500 							/* Total number of elements. */
#define LARGE_WHOLE 50000 							/* Total number of elements. */
#define HUGE_WHOLE 1000000 							/* Total number of elements. */

/*
** Max Hurdles - No Cycles
** Formula: 3*N + 1
** Number of Hurdles: 
*/
#define SMALL_MAX_NO_CYCLES 833 					/* 2500 elements. */
#define LARGE_MAX_NO_CYCLES 16667 					/* 50002 elements. */

/*
** Many Hurdles - No Cycles
** Formula: 7*N + 1
** Number of Hurdles: 
*/
#define SMALL_MANY_NO_CYCLES 357 					/* 2500 elements. */
#define LARGE_MANY_NO_CYCLES 7143 					/* 50002 elements. */
#define HUGE_MANY_NO_CYCLES 142857 					/* 1000000 elements. */

/*
** Many Hurdles - Cycles
** Formula: 7*N + 5
** Number of Hurdles: 
*/
#define SMALL_MANY_CYCLES 357 						/* 2504 elements. */
#define LARGE_MANY_CYCLES 7143 						/* 50006 elements. */

/*
** Fully Nested
** Formula: 4*(N + 2) + 5
** Number of Hurdles: 1
*/
#define SMALL_FULLY_NESTED 622 						/* 2501 elements. */
#define LARGE_FULLY_NESTED 12497 					/* 50001 elements. */
#define HUGE_FULLY_NESTED 249997 					/* 1000001 elements. */

/*
** Increasing Nests
** Formula: 2*H^2 + 10*H + 1
** Number of Hurdles: H
*/
#define SMALL_INCREASING_NESTS 33 					/* 2509 elements. */
#define LARGE_INCREASING_NESTS 156 					/* 50233 elements. */

/*
** Progressively Larger Hurdles
** Formula: 0,5*N^2 + 2,5*N + 5
** Number of Hurdles: 
*/
#define SMALL_PROGRESSIVE 69 						/* 2558 elements. */
#define LARGE_PROGRESSIVE 315 						/* 50405 elements. */

/*
** Progressively Larger Hurdles - No Cycles
** Formula: 0,5*N^2 + 2,5*N + 1
** Number of Hurdles: 
*/
#define SMALL_PROGRESSIVE_NO_CYCLES 69 				/* 2554 elements. */
#define LARGE_PROGRESSIVE_NO_CYCLES 315 			/* 50401 elements. */
#define HUGE_PROGRESSIVE_NO_CYCLES 1500 			/* 1128751 elements. */

/*
** Shell Progressively Larger
** Formula: 0,5*H^2 + 6,5*H + 1
** Number of Hurdles: H
*/
#define SMALL_SHELL_PROGRESSIVE 65 					/* 2536 elements. */
#define LARGE_SHELL_PROGRESSIVE 310 				/* 50066 elements. */

/*
** Shell Components
** Formula: 30*N + 1
** Number of Hurdles: 
*/
#define SMALL_SHELL_COMPONENTS 84 					/* 2521 elements. */
#define LARGE_SHELL_COMPONENTS 1667 				/* 50011 elements. */

/*
** Shell Heavy Nested
** Formula: 0,5*H^2 + 34,5*H + 1
** Number of Hurdles: H
*/
#define SMALL_SHELL_HEAVY_NESTED 45 				/* 2566 elements. */
#define LARGE_SHELL_HEAVY_NESTED 284 				/* 50127 elements. */

/*
** Shell Heavy Nested Components
** Formula: 93*C + 1
** Number of Hurdles: 
*/
#define SMALL_SHELL_HEAVY_NESTED_COMPONENTS 27 		/* 2512 elements. */
#define LARGE_SHELL_HEAVY_NESTED_COMPONENTS 538 	/* 50035 elements. */
#define HUGE_SHELL_HEAVY_NESTED_COMPONENTS 10753 	/* 1000030 elements. */

/*
** Divide and Conquer - Algorithm 1
** Formula: 2^{2*R - 1} + 5*2^R + 3*2^{R-1} - 3
** Number of Hurdles: 2^R
*/
#define SMALL_D_C_1 6 								/* 2461 elements. */
#define LARGE_D_C_1 8 								/* 34429 elements. */
#define HUGE_D_C_1 10 								/* 530941 elements. */

/*
** Divide and Conquer - Algorithm 2
** Formula: 3*2^R + 16*2^{R-1} - 3
** Number of Hurdles: 2^R
*/
#define SMALL_D_C_2 8 								/* 2813 elements. */
#define LARGE_D_C_2 12 								/* 45053 elements. */
#define HUGE_D_C_2 17 								/* 1441789 elements. */

/*********************************************************/

	/**********************************\
	* GENERATOR-FUNCTIONS-DECLARATIONS *
	\**********************************/

int		Generate_Example (char *);
int		Generate_Inverse_Permutation (char *, const int);
int		Whole_Sequence (char *, const int);
int		Max_Hurdles_No_Cycles (char *, const int);
int		Many_Hurdles_No_Cycles (char *, const int);
int 	Many_Hurdles_Cycles (char *, const int);
int 	Fully_Nested (char *, const int);
int 	Increasing_Nests (char *, const int);
int 	Progressively_Larger_Hurdles (char *, const int);
int 	Progressively_Larger_Hurdles_No_Cycles (char *, const int);

int 	Shell_Progressively_Larger (char *, const int);
int 	Shell_Components (char *, const int);
int 	Shell_Heavy_Nested (char *, const int);
int 	Shell_Heavy_Nested_Components (char *, const int);
int		Divide_and_Conquer_1 (char *, const int);
void	D_C_1 (int *, const int, const int, const int, const int, const int);
int		Divide_and_Conquer_2 (char *, const int);
void	D_C_2 (int *, const int, const int, const int);

int		Print_Hurdle (FILE *, int, int);
int		Generate_Hurdle (int *, const int, const int, const int);

/*********************************************************/



/*********************************************************/

int		main (void)
{
	char	filename [10];
	int		temp;


	/*********\
	* Example *
	\*********/

	/* Set file name. */
	strcpy (filename, "perm01.in");
	filename [9] = '\0';

	fprintf (stdout, "Generating permutation  1. Please wait ... ");
	fflush (stdout);
	temp = Generate_Example (filename);
	if (temp == 1)
		fprintf (stdout, "DONE!\n");
	else {
		fprintf (stdout, "\nError while creating file. Exiting ...\n");
		return	-1;		/* Indicate error. */
	}


	/******************\
	* Inversely Sorted *
	\******************/

	/* Set file name. */
	filename [5] = '2';

	fprintf (stdout, "Generating permutation  2. Please wait ... ");
	fflush (stdout);
	temp = Generate_Inverse_Permutation (filename, SMALL_INVERSE);
	if (temp == 1)
		fprintf (stdout, "DONE!\n");
	else {
		fprintf (stdout, "\nError while creating file. Exiting ...\n");
		return	-1;		/* Indicate error. */
	}


	/****************\
	* Whole Sequence *
	\****************/

	/* Set file name. */
	filename [5] = '3';

	fprintf (stdout, "Generating permutation  3. Please wait ... ");
	fflush (stdout);
	temp = Whole_Sequence (filename, SMALL_WHOLE);
	if (temp == 1)
		fprintf (stdout, "DONE!\n");
	else {
		fprintf (stdout, "\nError while creating file. Exiting ...\n");
		return	-1;		/* Indicate error. */
	}


	/*************************\
	* Max Hurdles - No Cycles *
	\*************************/

	/* Set file name. */
	filename [5] = '4';

	fprintf (stdout, "Generating permutation  4. Please wait ... ");
	fflush (stdout);
	temp = Max_Hurdles_No_Cycles (filename, SMALL_MAX_NO_CYCLES);
	if (temp == 1)
		fprintf (stdout, "DONE!\n");
	else {
		fprintf (stdout, "\nError while creating file. Exiting ...\n");
		return	-1;		/* Indicate error. */
	}


	/**************************\
	* Many Hurdles - No Cycles *
	\**************************/

	/* Set file name. */
	filename [5] = '5';

	fprintf (stdout, "Generating permutation  5. Please wait ... ");
	fflush (stdout);
	temp = Many_Hurdles_No_Cycles (filename, SMALL_MANY_NO_CYCLES);
	if (temp == 1)
		fprintf (stdout, "DONE!\n");
	else {
		fprintf (stdout, "\nError while creating file. Exiting ...\n");
		return	-1;		/* Indicate error. */
	}


	/***********************\
	* Many Hurdles - Cycles *
	\***********************/

	/* Set file name. */
	filename [5] = '6';

	fprintf (stdout, "Generating permutation  6. Please wait ... ");
	fflush (stdout);
	temp = Many_Hurdles_Cycles (filename, SMALL_MANY_CYCLES);
	if (temp == 1)
		fprintf (stdout, "DONE!\n");
	else {
		fprintf (stdout, "\nError while creating file. Exiting ...\n");
		return	-1;		/* Indicate error. */
	}


	/**************\
	* Fully_Nested *
	\**************/

	/* Set file name. */
	filename [5] = '7';

	fprintf (stdout, "Generating permutation  7. Please wait ... ");
	fflush (stdout);
	temp = Fully_Nested (filename, SMALL_FULLY_NESTED);
	if (temp == 1)
		fprintf (stdout, "DONE!\n");
	else {
		fprintf (stdout, "\nError while creating file. Exiting ...\n");
		return	-1;		/* Indicate error. */
	}


	/******************\
	* Increasing Nests *
	\******************/

	/* Set file name. */
	filename [5] = '8';

	fprintf (stdout, "Generating permutation  8. Please wait ... ");
	fflush (stdout);
	temp = Increasing_Nests (filename, SMALL_INCREASING_NESTS);
	if (temp == 1)
		fprintf (stdout, "DONE!\n");
	else {
		fprintf (stdout, "\nError while creating file. Exiting ...\n");
		return	-1;		/* Indicate error. */
	}


	/********************\
	* Progressive Cycles *
	\********************/

	/* Set file name. */
	filename [5] = '9';

	fprintf (stdout, "Generating permutation  9. Please wait ... ");
	fflush (stdout);
	temp = Progressively_Larger_Hurdles (filename, SMALL_PROGRESSIVE);
	if (temp == 1)
		fprintf (stdout, "DONE!\n");
	else {
		fprintf (stdout, "\nError while creating file. Exiting ...\n");
		return	-1;		/* Indicate error. */
	}


	/***********************\
	* Progressive No Cycles *
	\***********************/

	/* Set file name. */
	filename [4] = '1';
	filename [5] = '0';

	fprintf (stdout, "Generating permutation 10. Please wait ... ");
	fflush (stdout);
	temp = Progressively_Larger_Hurdles_No_Cycles (filename, SMALL_PROGRESSIVE_NO_CYCLES);
	if (temp == 1)
		fprintf (stdout, "DONE!\n");
	else {
		fprintf (stdout, "\nError while creating file. Exiting ...\n");
		return	-1;		/* Indicate error. */
	}


	/************************************\
	* Shell Progressively Larger Hurdles *
	\************************************/

	/* Set file name. */
	filename [5] = '1';

	fprintf (stdout, "Generating permutation 11. Please wait ... ");
	fflush (stdout);

	temp = Shell_Progressively_Larger (filename, SMALL_SHELL_PROGRESSIVE);
	if (temp == 1)
		fprintf (stdout, "DONE!\n");
	else {
		fprintf (stdout, "\nError while creating file. Exiting ...\n");
		return	-1;		/* Indicate error. */
	}


	/******************\
	* Shell Components *
	\******************/

	/* Set file name. */
	filename [5] = '2';

	fprintf (stdout, "Generating permutation 12. Please wait ... ");
	fflush (stdout);

	temp = Shell_Components (filename, SMALL_SHELL_COMPONENTS);
	if (temp == 1)
		fprintf (stdout, "DONE!\n");
	else {
		fprintf (stdout, "\nError while creating file. Exiting ...\n");
		return	-1;		/* Indicate error. */
	}


	/********************\
	* Shell Heavy Nested *
	\********************/

	/* Set file name. */
	filename [5] = '3';

	fprintf (stdout, "Generating permutation 13. Please wait ... ");
	fflush (stdout);

	temp = Shell_Heavy_Nested (filename, SMALL_SHELL_HEAVY_NESTED);
	if (temp == 1)
		fprintf (stdout, "DONE!\n");
	else {
		fprintf (stdout, "\nError while creating file. Exiting ...\n");
		return	-1;		/* Indicate error. */
	}


	/*******************************\
	* Shell Heavy Nested Components *
	\*******************************/

	/* Set file name. */
	filename [5] = '4';

	fprintf (stdout, "Generating permutation 14. Please wait ... ");
	fflush (stdout);

	temp = Shell_Heavy_Nested_Components (filename, SMALL_SHELL_HEAVY_NESTED_COMPONENTS);
	if (temp == 1)
		fprintf (stdout, "DONE!\n");
	else {
		fprintf (stdout, "\nError while creating file. Exiting ...\n");
		return	-1;		/* Indicate error. */
	}


	/***********************\
	* Divide and Conquer #1 *
	\***********************/

	/* Set file name. */
	filename [5] = '5';

	fprintf (stdout, "Generating permutation 15. Please wait ... ");
	fflush (stdout);

	temp = Divide_and_Conquer_1 (filename, SMALL_D_C_1);
	if (temp == 1)
		fprintf (stdout, "DONE!\n");
	else {
		fprintf (stdout, "\nError while creating file. Exiting ...\n");
		return	-1;		/* Indicate error. */
	}


	/***********************\
	* Divide and Conquer #2 *
	\***********************/

	/* Set file name. */
	filename [5] = '6';

	fprintf (stdout, "Generating permutation 16. Please wait ... ");
	fflush (stdout);

	temp = Divide_and_Conquer_2 (filename, SMALL_D_C_2);
	if (temp == 1)
		fprintf (stdout, "DONE!\n");
	else {
		fprintf (stdout, "\nError while creating file. Exiting ...\n");
		return	-1;		/* Indicate error. */
	}



	/******************\
	* Inversely Sorted *
	\******************/

	/* Set file name. */
	filename [5] = '7';

	fprintf (stdout, "Generating permutation 17. Please wait ... ");
	fflush (stdout);
	temp = Generate_Inverse_Permutation (filename, LARGE_INVERSE);
	if (temp == 1)
		fprintf (stdout, "DONE!\n");
	else {
		fprintf (stdout, "\nError while creating file. Exiting ...\n");
		return	-1;		/* Indicate error. */
	}


	/****************\
	* Whole Sequence *
	\****************/

	/* Set file name. */
	filename [5] = '8';

	fprintf (stdout, "Generating permutation 18. Please wait ... ");
	fflush (stdout);
	temp = Whole_Sequence (filename, LARGE_WHOLE);
	if (temp == 1)
		fprintf (stdout, "DONE!\n");
	else {
		fprintf (stdout, "\nError while creating file. Exiting ...\n");
		return	-1;		/* Indicate error. */
	}


	/*************************\
	* Max Hurdles - No Cycles *
	\*************************/

	/* Set file name. */
	filename [5] = '9';

	fprintf (stdout, "Generating permutation 19. Please wait ... ");
	fflush (stdout);
	temp = Max_Hurdles_No_Cycles (filename, LARGE_MAX_NO_CYCLES);
	if (temp == 1)
		fprintf (stdout, "DONE!\n");
	else {
		fprintf (stdout, "\nError while creating file. Exiting ...\n");
		return	-1;		/* Indicate error. */
	}


	/**************************\
	* Many Hurdles - No Cycles *
	\**************************/

	/* Set file name. */
	filename [4] = '2';
	filename [5] = '0';

	fprintf (stdout, "Generating permutation 20. Please wait ... ");
	fflush (stdout);
	temp = Many_Hurdles_No_Cycles (filename, LARGE_MANY_NO_CYCLES);
	if (temp == 1)
		fprintf (stdout, "DONE!\n");
	else {
		fprintf (stdout, "\nError while creating file. Exiting ...\n");
		return	-1;		/* Indicate error. */
	}


	/***********************\
	* Many Hurdles - Cycles *
	\***********************/

	/* Set file name. */
	filename [5] = '1';

	fprintf (stdout, "Generating permutation 21. Please wait ... ");
	fflush (stdout);
	temp = Many_Hurdles_Cycles (filename, LARGE_MANY_CYCLES);
	if (temp == 1)
		fprintf (stdout, "DONE!\n");
	else {
		fprintf (stdout, "\nError while creating file. Exiting ...\n");
		return	-1;		/* Indicate error. */
	}


	/**************\
	* Fully_Nested *
	\**************/

	/* Set file name. */
	filename [5] = '2';

	fprintf (stdout, "Generating permutation 22. Please wait ... ");
	fflush (stdout);
	temp = Fully_Nested (filename, LARGE_FULLY_NESTED);
	if (temp == 1)
		fprintf (stdout, "DONE!\n");
	else {
		fprintf (stdout, "\nError while creating file. Exiting ...\n");
		return	-1;		/* Indicate error. */
	}


	/******************\
	* Increasing Nests *
	\******************/

	/* Set file name. */
	filename [5] = '3';

	fprintf (stdout, "Generating permutation 23. Please wait ... ");
	fflush (stdout);
	temp = Increasing_Nests (filename, LARGE_INCREASING_NESTS);
	if (temp == 1)
		fprintf (stdout, "DONE!\n");
	else {
		fprintf (stdout, "\nError while creating file. Exiting ...\n");
		return	-1;		/* Indicate error. */
	}


	/********************\
	* Progressive Cycles *
	\********************/

	/* Set file name. */
	filename [5] = '4';

	fprintf (stdout, "Generating permutation 24. Please wait ... ");
	fflush (stdout);
	temp = Progressively_Larger_Hurdles (filename, LARGE_PROGRESSIVE);
	if (temp == 1)
		fprintf (stdout, "DONE!\n");
	else {
		fprintf (stdout, "\nError while creating file. Exiting ...\n");
		return	-1;		/* Indicate error. */
	}


	/***********************\
	* Progressive No Cycles *
	\***********************/

	/* Set file name. */
	filename [5] = '5';

	fprintf (stdout, "Generating permutation 25. Please wait ... ");
	fflush (stdout);
	temp = Progressively_Larger_Hurdles_No_Cycles (filename, LARGE_PROGRESSIVE_NO_CYCLES);
	if (temp == 1)
		fprintf (stdout, "DONE!\n");
	else {
		fprintf (stdout, "\nError while creating file. Exiting ...\n");
		return	-1;		/* Indicate error. */
	}


	/************************************\
	* Shell Progressively Larger Hurdles *
	\************************************/

	/* Set file name. */
	filename [5] = '6';

	fprintf (stdout, "Generating permutation 26. Please wait ... ");
	fflush (stdout);

	temp = Shell_Progressively_Larger (filename, LARGE_SHELL_PROGRESSIVE);
	if (temp == 1)
		fprintf (stdout, "DONE!\n");
	else {
		fprintf (stdout, "\nError while creating file. Exiting ...\n");
		return	-1;		/* Indicate error. */
	}


	/******************\
	* Shell Components *
	\******************/

	/* Set file name. */
	filename [5] = '7';

	fprintf (stdout, "Generating permutation 27. Please wait ... ");
	fflush (stdout);

	temp = Shell_Components (filename, LARGE_SHELL_COMPONENTS);
	if (temp == 1)
		fprintf (stdout, "DONE!\n");
	else {
		fprintf (stdout, "\nError while creating file. Exiting ...\n");
		return	-1;		/* Indicate error. */
	}


	/********************\
	* Shell Heavy Nested *
	\********************/

	/* Set file name. */
	filename [5] = '8';

	fprintf (stdout, "Generating permutation 28. Please wait ... ");
	fflush (stdout);

	temp = Shell_Heavy_Nested (filename, LARGE_SHELL_HEAVY_NESTED);
	if (temp == 1)
		fprintf (stdout, "DONE!\n");
	else {
		fprintf (stdout, "\nError while creating file. Exiting ...\n");
		return	-1;		/* Indicate error. */
	}


	/*******************************\
	* Shell Heavy Nested Components *
	\*******************************/

	/* Set file name. */
	filename [5] = '9';

	fprintf (stdout, "Generating permutation 29. Please wait ... ");
	fflush (stdout);

	temp = Shell_Heavy_Nested_Components (filename, LARGE_SHELL_HEAVY_NESTED_COMPONENTS);
	if (temp == 1)
		fprintf (stdout, "DONE!\n");
	else {
		fprintf (stdout, "\nError while creating file. Exiting ...\n");
		return	-1;		/* Indicate error. */
	}


	/***********************\
	* Divide and Conquer #1 *
	\***********************/

	/* Set file name. */
	filename [4] = '3';
	filename [5] = '0';

	fprintf (stdout, "Generating permutation 30. Please wait ... ");
	fflush (stdout);

	temp = Divide_and_Conquer_1 (filename, LARGE_D_C_1);
	if (temp == 1)
		fprintf (stdout, "DONE!\n");
	else {
		fprintf (stdout, "\nError while creating file. Exiting ...\n");
		return	-1;		/* Indicate error. */
	}


	/***********************\
	* Divide and Conquer #2 *
	\***********************/

	/* Set file name. */
	filename [5] = '1';

	fprintf (stdout, "Generating permutation 31. Please wait ... ");
	fflush (stdout);

	temp = Divide_and_Conquer_2 (filename, LARGE_D_C_2);
	if (temp == 1)
		fprintf (stdout, "DONE!\n");
	else {
		fprintf (stdout, "\nError while creating file. Exiting ...\n");
		return	-1;		/* Indicate error. */
	}


	/****************\
	* Whole Sequence *
	\****************/

	/* Set file name. */
	filename [5] = '2';

	fprintf (stdout, "Generating permutation 32. Please wait ... ");
	fflush (stdout);
	temp = Whole_Sequence (filename, HUGE_WHOLE);
	if (temp == 1)
		fprintf (stdout, "DONE!\n");
	else {
		fprintf (stdout, "\nError while creating file. Exiting ...\n");
		return	-1;		/* Indicate error. */
	}


	/**************************\
	* Many Hurdles - No Cycles *
	\**************************/

	/* Set file name. */
	filename [5] = '3';

	fprintf (stdout, "Generating permutation 33. Please wait ... ");
	fflush (stdout);
	temp = Many_Hurdles_No_Cycles (filename, HUGE_MANY_NO_CYCLES);
	if (temp == 1)
		fprintf (stdout, "DONE!\n");
	else {
		fprintf (stdout, "\nError while creating file. Exiting ...\n");
		return	-1;		/* Indicate error. */
	}


	/**************\
	* Fully_Nested *
	\**************/

	/* Set file name. */
	filename [5] = '4';

	fprintf (stdout, "Generating permutation 34. Please wait ... ");
	fflush (stdout);
	temp = Fully_Nested (filename, HUGE_FULLY_NESTED);
	if (temp == 1)
		fprintf (stdout, "DONE!\n");
	else {
		fprintf (stdout, "\nError while creating file. Exiting ...\n");
		return	-1;		/* Indicate error. */
	}


	/***********************\
	* Progressive No Cycles *
	\***********************/

	/* Set file name. */
	filename [5] = '5';

	fprintf (stdout, "Generating permutation 35. Please wait ... ");
	fflush (stdout);
	temp = Progressively_Larger_Hurdles_No_Cycles (filename, HUGE_PROGRESSIVE_NO_CYCLES);
	if (temp == 1)
		fprintf (stdout, "DONE!\n");
	else {
		fprintf (stdout, "\nError while creating file. Exiting ...\n");
		return	-1;		/* Indicate error. */
	}


	/*******************************\
	* Shell Heavy Nested Components *
	\*******************************/

	/* Set file name. */
	filename [5] = '6';

	fprintf (stdout, "Generating permutation 36. Please wait ... ");
	fflush (stdout);

	temp = Shell_Heavy_Nested_Components (filename, HUGE_SHELL_HEAVY_NESTED_COMPONENTS);
	if (temp == 1)
		fprintf (stdout, "DONE!\n");
	else {
		fprintf (stdout, "\nError while creating file. Exiting ...\n");
		return	-1;		/* Indicate error. */
	}


	/***********************\
	* Divide and Conquer #1 *
	\***********************/

	/* Set file name. */
	filename [5] = '7';

	fprintf (stdout, "Generating permutation 37. Please wait ... ");
	fflush (stdout);

	temp = Divide_and_Conquer_1 (filename, HUGE_D_C_1);
	if (temp == 1)
		fprintf (stdout, "DONE!\n");
	else {
		fprintf (stdout, "\nError while creating file. Exiting ...\n");
		return	-1;		/* Indicate error. */
	}


	/***********************\
	* Divide and Conquer #2 *
	\***********************/

	/* Set file name. */
	filename [5] = '8';

	fprintf (stdout, "Generating permutation 38. Please wait ... ");
	fflush (stdout);

	temp = Divide_and_Conquer_2 (filename, HUGE_D_C_2);
	if (temp == 1)
		fprintf (stdout, "DONE!\n");
	else {
		fprintf (stdout, "\nError while creating file. Exiting ...\n");
		return	-1;		/* Indicate error. */
	}



	return 	0; 	/* Successful generation of input files. */
}



/*********************************************************/

/*********************************************************/

/* Generate example. */
/* <0 3 5 4 6 2 1 7> */

int		Generate_Example (char * filename)
{
	FILE *	fd;

	fd = fopen (filename, "w");

	if (fd == (FILE *) NULL)
		return	-1;		/* Indicate error. */

	fprintf (fd, "8\n");
	fprintf (fd, "0\n");
	fprintf (fd, "3\n");
	fprintf (fd, "5\n");
	fprintf (fd, "4\n");
	fprintf (fd, "6\n");
	fprintf (fd, "2\n");
	fprintf (fd, "1\n");
	fprintf (fd, "7\n");

	fclose (fd);

	return	1;	/* File creation was successful. */
}

/*********************************************************/

/*
** Inversely sorted permutation.
** <0 N (N - 1) (N - 2) ... 3 2 1 (N + 1)>
*/

int		Generate_Inverse_Permutation (char * filename, const int size)
{
	FILE * 			fd;
	register int 	i;

	fd = fopen (filename, "w");

	if (fd == (FILE *) NULL)
		return	-1;		/* Indicate error. */

	fprintf (fd, "%d\n", size);
	fprintf (fd, "0\n");
	for (i = size - 2; i >= 1; i--)
		fprintf (fd, "%d\n", i);
	fprintf (fd, "%d\n", size - 1);

	fclose (fd);

	return	1;	/* File creation was successful. */
}

/*********************************************************/

/*
** The whole sequence is a hurdle,
** but it is not inversely sorted.
*/

int		Whole_Sequence (char * filename, const int size)
{
	FILE * 			fd;

	fd = fopen (filename, "w");

	if (fd == (FILE *) NULL)
		return	-1;		/* Indicate error. */

	fprintf (fd, "%d\n", size);
	fprintf (fd, "0\n");
	Print_Hurdle (fd, 0, size);

	fclose (fd);

	return	1;	/* File creation was successful. */

}

/*********************************************************/

/*
** Generate as many hurdles as possible. No cyclic hurdles.
** <0 2 1 3 5 4 6 ... (N - 2) N (N - 1) (N + 1)>
*/

int		Max_Hurdles_No_Cycles (char * filename, const int num_hurdles)
{
	FILE * 			fd;
	register int 	i;

	fd = fopen (filename, "w");

	if (fd == (FILE *) NULL)
		return	-1;		/* Indicate error. */

	fprintf (fd, "%d\n", num_hurdles * 3 + 1);
	fprintf (fd, "0\n");
	for (i = 1; i < num_hurdles * 3; i++) {
		if (i % 3 == 1)
			fprintf (fd, "%d\n", i + 1);
		else if (i % 3 == 2)
			fprintf (fd, "%d\n", i - 1);
		else
			fprintf (fd, "%d\n", i);
	}
	fprintf (fd, "%d\n", num_hurdles * 3);

	fclose (fd);

	return	1;	/* File creation was successful. */
}

/*********************************************************/

/*
** Generate a sequence of 5-element hurdles accompanied by 4-element hurdles.
** e.g.:
** <0 3 2 1 4 6 5 7 ... (N - 6) (N - 3) (N - 4) (N - 5) (N - 2) N (N - 1) (N + 1)>
**  ^       ^     ^        ^                               ^                 ^
*/

int		Many_Hurdles_No_Cycles (char * filename, const int repetitions)
{
	FILE * 			fd;
	register int 	i;

	fd = fopen (filename, "w");

	if (fd == (FILE *) NULL)
		return	-1;		/* Indicate error. */

	fprintf (fd, "%d\n", repetitions * 7 + 1);
	for (i = 0; i < repetitions * 7 + 1; i++) {
		if ((i % 7 == 0) || (i % 7 == 2) || (i % 7 == 4))
			fprintf (fd, "%d\n", i);
		else if (i % 7 == 1)
			fprintf (fd, "%d\n", i + 2);
		else if (i % 7 == 3)
			fprintf (fd, "%d\n", i - 2);
		else if (i % 7 == 5)
			fprintf (fd, "%d\n", i + 1);
		else
			fprintf (fd, "%d\n", i - 1);	/* (i % 7 == 6) */
	}

	fclose (fd);

	return	1;	/* File creation was successful. */
}

/*********************************************************/

/*
** Generate a sequence of 4-element hurdles accompanied by 5-element hurdles.
** Moreover, allow a cyclic hurdle.
** e.g.:
** <0 3 5 4 6 9 8 7 10 ... (N-7) (N-5) (N-6) (N-4) (N-1) (N-2) (N-3) N 2 1 (N+1)>
**    ^     ^        ^       ^                 ^                     ^
*/

int 	Many_Hurdles_Cycles (char * filename, const int repetitions)
{
	FILE * 			fd;
	register int 	i;
	register int 	current;

	fd = fopen (filename, "w");

	if (fd == (FILE *) NULL)
		return	-1;		/* Indicate error. */

	fprintf (fd, "%d\n", repetitions * 7 + 5);
	fprintf (fd, "0\n");
	fprintf (fd, "3\n");
	for (i = 0; i < repetitions; i++) {
		current = 7 * i + 5;
		fprintf (fd, "%d\n", current);
		current--;
		fprintf (fd, "%d\n", current);
		current += 2;
		fprintf (fd, "%d\n", current);
		current += 3;
		fprintf (fd, "%d\n", current);
		current--;
		fprintf (fd, "%d\n", current);
		current--;
		fprintf (fd, "%d\n", current);
		current += 3;
		fprintf (fd, "%d\n", current);
	}
	fprintf (fd, "2\n");
	fprintf (fd, "1\n");
	fprintf (fd, "%d\n", 7 * repetitions + 4);

	fclose (fd);

	return 	1; 	/* File creation was successful. */
}

/*********************************************************/

/*
** Generate a hurdle that is nested.
** Allow a cyclic hurdle.
** e.g.:
** <0 3 6 8 7 9 5 4 10 2 1 11>  {1 Nest}
**    ^ ^     ^      ^
**    ^ --- NEST --- ^          { 2 Hurdles: (2:5), (8:1). }
** General formula:
** <0 3 6 9 ... (3*Nests) (3*[Nests+1]) (3*[Nests+1]+2) (3*[Nests+1]+1) (3*[Nests+2]) (3*[Nests+1]-1) (3*[Nests+1]-2) (3*[Nests+2] + 1) ... 2 1 (3*[Nests+2]+Nests+1) >
**                              ^                                              ^
*/

int 	Fully_Nested (char * filename, const int nests)
{
	FILE * 			fd;
	register int 	i;
	register int 	increasing;

	fd = fopen (filename, "w");

	if (fd == (FILE *) NULL)
		return	-1;		/* Indicate error. */

	fprintf (fd, "%d\n", (nests + 2) * 4 + 5);

	for (i = 0; i < nests + 2; i++)
		fprintf (fd, "%d\n", i * 3);

	i--;
	/* Generate a 9-element Hurdle. */
	Print_Hurdle (fd, 3 * i, 9);

	increasing = 3 * (i + 1) + 6;
	for (i = nests + 1; i >= 1; i--) {
		fprintf (fd, "%d\n", i * 3 - 1);
		fprintf (fd, "%d\n", i * 3 - 2);
		fprintf (fd, "%d\n", increasing);
		increasing++;
	}

	fclose (fd);

	return 	1; 	/* File creation was successful. */
}

/*********************************************************/

/*
** Generate 9-element hurdles, but with increasing
** number of nests covering each hurdle.
*/

int 	Increasing_Nests (char * filename, const int hurdles)
{
	FILE * 			fd;
	register int 	i;
	register int	j;
	register int 	increasing;
	register int	decreasing;

	fd = fopen (filename, "w");

	if (fd == (FILE *) NULL)
		return	-1;		/* Indicate error. */

	fprintf (fd, "%d\n", 2 * hurdles * (hurdles + 5) + 1);

	/* Initializations. */
	increasing = 0;

	for (i = 0; i < hurdles; i++) {
		/* Open (i + 1) nests. */
		for (j = 1; j <= i + 1; j++) {
			fprintf (fd, "%d\n", increasing);
			increasing += 3;
		}
		decreasing = increasing - 1;
		/* Print a 9-element hurdle. */
		fprintf (fd, "%d\n", increasing);
		Print_Hurdle (fd, increasing, 9);
		increasing += 9;
		fprintf (fd, "%d\n", decreasing);
		decreasing--;
		fprintf (fd, "%d\n", decreasing);
		decreasing--;
		for (j = 1; j <= i; j++) {
			fprintf (fd, "%d\n", increasing);
			increasing++;
			decreasing--;
			fprintf (fd, "%d\n", decreasing);
			decreasing--;
			fprintf (fd, "%d\n", decreasing);
			decreasing--;
		}
	}

	fprintf (fd, "%d\n", increasing);

	fclose (fd);

	return 	1; 	/* File creation was successful. */
}

/*********************************************************/

/*
** Generate a sequence of progressively length-increasing hurdles.
** Moreover, allow cyclic hurdles.
** e.g.:
** <0 3 5 4 6 9 8 7 10  ...  (N-I+1) (N-1) (N-2) (N-3) ... (N-I+3) (N-I+2) N 2 1 (N+1)>
**    ^     ^        ^          ^                                          ^
**      -4-   --5--     ...      ---------------- I terms -----------------
*/

int 	Progressively_Larger_Hurdles (char * filename, const int interior_hurdles)
{
	FILE * 			fd;
	register int 	i;
	register int 	N;
	register int 	mark;
	register int 	j;
	register int 	current;

	fd = fopen (filename, "w");

	if (fd == (FILE *) NULL)
		return	-1;		/* Indicate error. */

	N = (interior_hurdles * (interior_hurdles + 1)) / 2 + 2 * interior_hurdles + 5;

	fprintf (fd, "%d\n", N);
	fprintf (fd, "0\n");
	fprintf (fd, "3\n");
	mark = 3;
	for (i = 0; i < interior_hurdles; i++) {
		current = mark + 3 + i - 1;
		for (j = 1; j < 3 + i; j++) {
			fprintf (fd, "%d\n", current);
			current--;
		}
		mark += 3 + i;
		fprintf (fd, "%d\n", mark);
	}
	fprintf (fd, "2\n");
	fprintf (fd, "1\n");
	fprintf (fd, "%d\n", N - 1);

	fclose (fd);

	return 	1; 	/* File creation was successful. */
}

/*********************************************************/

/*
** Generate a sequence of progressively length-increasing hurdles.
** Do NOT allow cyclic hurdles.
** e.g.:
** <0 2 1 3 6 5 4 7 10  ...  (N-I+1) N (N-1) (N-2) ... (N-I+3) (N-I+2) (N+1)>
**  ^     ^       ^          ^                                           ^
**    -4-   --5--       ...   ----------------- I terms -----------------
*/

int 	Progressively_Larger_Hurdles_No_Cycles (char * filename, const int hurdles)
{
	FILE * 			fd;
	register int 	i;
	register int 	N;
	register int 	mark;
	register int 	j;
	register int 	current;

	fd = fopen (filename, "w");

	if (fd == (FILE *) NULL)
		return	-1;		/* Indicate error. */

	N = (hurdles * (hurdles + 1)) / 2 + 2 * hurdles + 1;

	fprintf (fd, "%d\n", N);
	fprintf (fd, "0\n");
	mark = 0;
	for (i = 0; i < hurdles; i++) {
		current = mark + 3 + i - 1;
		for (j = 1; j < 3 + i; j++) {
			fprintf (fd, "%d\n", current);
			current--;
		}
		mark += 3 + i;
		fprintf (fd, "%d\n", mark);
	}

	fclose (fd);

	return 	1; 	/* File creation was successful. */
}

/*********************************************************/

/*
**         - Hurdle -       --- Hurdle ---       ----- Hurdle ----
** <0 3 6 {9 11 10 12} 8 7 {13 16 15 14 17} 5 4 {18 20 19 22 21 23} 2 1 24>
**  ^ ^ ^              ^ ^                  ^ ^                     ^ ^
**  ^ ^ ^-----Shell----^-^                  ^ ^                     ^ ^
**  ^ ^-------Shell-------------------------^-^                     ^ ^
**  ^---------Shell-------------------------------------------------^-^
*/

int 	Shell_Progressively_Larger (char * filename, const int hurdles)
{
	FILE * 			fd;
	register int 	i;
	register int 	elements;
	register int 	counter;
	register int 	hurdle_starter;
	register int 	closing_starter;

	fd = fopen (filename, "w");

	if (fd == (FILE *) NULL)
		return	-1;		/* Indicate error. */

	/* Calculate total number of elements. */
	elements = 3 * hurdles + 1;
	elements += (hurdles + 3) * (hurdles + 4) / 2;
	elements -= 6;

	fprintf (fd, "%d\n", elements);

	fprintf (fd, "%d\n", 0);
	for (i = 1; i < hurdles; i++)
		fprintf (fd, "%d\n", 3 * i);
	hurdle_starter = 3 * i;
	closing_starter = hurdle_starter - 1;

	for (counter = 0; counter < hurdles; counter++) {
		/* Generate hurdle of 4 + counter elements. */
		elements = 4 + counter;
		fprintf (fd, "%d\n", hurdle_starter);
		Print_Hurdle (fd, hurdle_starter, elements);
		hurdle_starter += elements;
		/* Close shell. */
		fprintf (fd, "%d\n", closing_starter);
		closing_starter--;
		fprintf (fd, "%d\n", closing_starter);
		closing_starter -= 2;
	}

	/* Finish permutation. */
	fprintf (fd, "%d\n", hurdle_starter);


	fclose (fd);

	return 	1; 	/* File creation was successful. */
}

/*********************************************************/

/*
** Works like the previous Shell function, but
** forms components of 6, 7 and 8-element hurdles.
*/

int 	Shell_Components (char * filename, const int components)
{
	FILE * 			fd;
	register int 	i;
	register int 	elements;
	register int	hurdles;
	register int 	hurdle_starter;
	register int 	closing_starter;

	fd = fopen (filename, "w");

	if (fd == (FILE *) NULL)
		return	-1;		/* Indicate error. */

	/* Calculate total number of elements. */
	elements = 30 * components + 1;
	hurdles = 3 * components;

	fprintf (fd, "%d\n", elements);

	/* Initialize Shells. */
	fprintf (fd, "%d\n", 0);
	for (i = 1; i < hurdles; i++)
		fprintf (fd, "%d\n", 3 * i);
	hurdle_starter = 3 * i;
	closing_starter = hurdle_starter - 1;

	for (i = 0; i < hurdles; i++) {
		/* Generate hurdle of 4 + (i % 3) elements. */
		elements = 6 + (i % 3);
		fprintf (fd, "%d\n", hurdle_starter);
		Print_Hurdle (fd, hurdle_starter, elements);
		hurdle_starter += elements;
		/* Close shell. */
		fprintf (fd, "%d\n", closing_starter);
		closing_starter--;
		fprintf (fd, "%d\n", closing_starter);
		closing_starter -= 2;
	}

	/* Finish permutation. */
	fprintf (fd, "%d\n", hurdle_starter);


	fclose (fd);

	return 	1; 	/* File creation was successful. */
}

/*********************************************************/

/*
** All hurdles are nested (4 normal nests - 1 inverse nest) and between two different
** nests there exists a 3-fold shell.
** Finally, a 3-fold shell covers the entire permutation.
** (Progressively larger hurdles)
**
** e.g.:
**
**                          -  Hurdle -
** <0 3 6 9 12 17 16 18 21 {24 26 25 27} 23 22 28 20 19 29 15 30 14 13 31 11 10 32 8 7 33 5 4 34 2 1 35>
**  ^ ^ ^ ^  ^  ^  ^  ^  ^                ^  ^     ^  ^     ^     ^  ^     ^  ^    ^ ^    ^ ^    ^ ^
**  ^ ^ ^ ^  ^  ^  ^  ^  ^------Nest------^--^     ^  ^     ^     ^  ^     ^  ^    ^ ^    ^ ^    ^ ^
**  ^ ^ ^ ^  ^  ^  ^  ^---------Nest---------------^--^     ^     ^  ^     ^  ^    ^ ^    ^ ^    ^ ^
**  ^ ^ ^ ^  ^  ^--^----------Inverse_Nest------------------^     ^  ^     ^  ^    ^ ^    ^ ^    ^ ^
**  ^ ^ ^ ^  ^------------------Nest------------------------------^--^     ^  ^    ^ ^    ^ ^    ^ ^
**  ^ ^ ^ ^---------------------Nest---------------------------------------^--^    ^ ^    ^ ^    ^ ^
**  ^ ^ ^-------------------------------------Shell--------------------------------^-^    ^ ^    ^ ^
**  ^ ^---------------------------------------Shell---------------------------------------^-^    ^ ^
**  ^-----------------------------------------Shell----------------------------------------------^-^
*/

int 	Shell_Heavy_Nested (char * filename, const int hurdles)
{
	FILE * 			fd;
	register int 	i;
	register int 	elements;
	register int	counter;
	register int 	hurdle_starter;
	register int 	close_nest;
	register int	close_shell;

	fd = fopen (filename, "w");

	if (fd == (FILE *) NULL)
		return	-1;		/* Indicate error. */

	/* Calculate total number of elements. */
	elements = 31 * hurdles + 1;
	elements += (hurdles + 3) * (hurdles + 4) / 2;
	elements -= 6;

	fprintf (fd, "%d\n", elements);

	/* Open 3 Shells for each hurdle. */
	for (i = 0; i < 3 * hurdles; i++)
		fprintf (fd, "%d\n", 3 * i);
	hurdle_starter = 3 * i;
	close_shell = hurdle_starter - 1;

	for (counter = 0; counter < hurdles; counter++) {
		/* Open 5 Nests for each hurdle. */
		fprintf (fd, "%d\n", hurdle_starter);
		fprintf (fd, "%d\n", hurdle_starter + 3);
		fprintf (fd, "%d\n", hurdle_starter + 8);
		fprintf (fd, "%d\n", hurdle_starter + 7);
		fprintf (fd, "%d\n", hurdle_starter + 9);
		fprintf (fd, "%d\n", hurdle_starter + 12);
		hurdle_starter += 15;
		close_nest = hurdle_starter - 1;

		/* Generate hurdle of 4 + counter elements. */
		elements = 4 + counter;
		fprintf (fd, "%d\n", hurdle_starter);
		Print_Hurdle (fd, hurdle_starter, elements);
		hurdle_starter += elements;

		/* Close nests. */
		for (i = 1; i <= 2; i++) {
			fprintf (fd, "%d\n", close_nest);
			fprintf (fd, "%d\n", close_nest - 1);
			fprintf (fd, "%d\n", hurdle_starter);
			close_nest -= 3;
			hurdle_starter++;
		}
		close_nest -= 2;
		fprintf (fd, "%d\n", close_nest);
		fprintf (fd, "%d\n", hurdle_starter);
		close_nest--;
		hurdle_starter++;
		for (i = 1; i <= 2; i++) {
			fprintf (fd, "%d\n", close_nest);
			fprintf (fd, "%d\n", close_nest - 1);
			fprintf (fd, "%d\n", hurdle_starter);
			close_nest -= 3;
			hurdle_starter++;
		}

		/* Close shells. */
		for (i = 1; i <= 2; i++) {
			fprintf (fd, "%d\n", close_shell);
			fprintf (fd, "%d\n", close_shell - 1);
			fprintf (fd, "%d\n", hurdle_starter);
			close_shell -= 3;
			hurdle_starter++;
		}
		fprintf (fd, "%d\n", close_shell);
		fprintf (fd, "%d\n", close_shell - 1);
		close_shell -= 3;
	}

	/* Finish permutation. */
	fprintf (fd, "%d\n", hurdle_starter);


	fclose (fd);

	return 	1; 	/* File creation was successful. */
}

/*********************************************************/

/*
** All hurdles are nested (4 normal nests - 1 inverse nest) and between two different
** nests there exists a 3-fold shell.
** Finally, a 3-fold shell covers the entire permutation.
** (4, 6, 8, 10 and 12-element hurdles)
** Exactly like the previous function but allows components only of 4, 8 and 12-element hurdles.
*/

int 	Shell_Heavy_Nested_Components (char * filename, const int components)
{
	FILE * 			fd;
	register int 	i;
	register int	hurdle;
	register int	counter;
	register int 	hurdle_starter;
	register int 	close_nest;
	register int	close_shell;
	register int 	elements;

	fd = fopen (filename, "w");

	if (fd == (FILE *) NULL)
		return	-1;		/* Indicate error. */

	/* Calculate total number of elements. */
	elements = 93 * components + 1;

	fprintf (fd, "%d\n", elements);

	/* Open 1 Shell for each hurdle. */
	for (i = 0; i < 3 * components; i++)
		fprintf (fd, "%d\n", 3 * i);
	hurdle_starter = 3 * i;
	close_shell = hurdle_starter - 1;

	for (counter = 0; counter < components; counter++) {
		for (hurdle = 4; hurdle <= 12; hurdle += 4) {
			/* Open 5 Nests for each hurdle. */
			fprintf (fd, "%d\n", hurdle_starter);
			fprintf (fd, "%d\n", hurdle_starter + 3);
			fprintf (fd, "%d\n", hurdle_starter + 8);
			fprintf (fd, "%d\n", hurdle_starter + 7);
			fprintf (fd, "%d\n", hurdle_starter + 9);
			fprintf (fd, "%d\n", hurdle_starter + 12);
			hurdle_starter += 15;
			close_nest = hurdle_starter - 1;

			/* Generate hurdle of "hurdle" elements. */
			elements = hurdle;
			fprintf (fd, "%d\n", hurdle_starter);
			Print_Hurdle (fd, hurdle_starter, elements);
			hurdle_starter += elements;

			/* Close nests. */
			for (i = 1; i <= 2; i++) {
				fprintf (fd, "%d\n", close_nest);
				fprintf (fd, "%d\n", close_nest - 1);
				fprintf (fd, "%d\n", hurdle_starter);
				close_nest -= 3;
				hurdle_starter++;
			}
			close_nest -= 2;
			fprintf (fd, "%d\n", close_nest);
			fprintf (fd, "%d\n", hurdle_starter);
			close_nest--;
			hurdle_starter++;
			for (i = 1; i <= 2; i++) {
				fprintf (fd, "%d\n", close_nest);
				fprintf (fd, "%d\n", close_nest - 1);
				fprintf (fd, "%d\n", hurdle_starter);
				close_nest -= 3;
				hurdle_starter++;
			}

			/* Close Shell. */
			fprintf (fd, "%d\n", close_shell);
			fprintf (fd, "%d\n", close_shell - 1);
			close_shell -= 3;
		}
	}

	/* Finish permutation. */
	fprintf (fd, "%d\n", hurdle_starter);

	/* Close File. */
	fclose (fd);

	return 	1; 	/* File creation was successful. */
}

/*********************************************************/

int		Divide_and_Conquer_1 (char * filename, const int Recursion_Levels)
{
	FILE *			fd;
	register int	i;
	register int	Elements;
	register int	power;
	int	*			Array;

	/*
	** Groups of 4-elements        : 2^{Recursion_Levels} - 1.
	** Progressively larger hurdles: 2^{Recursion_Levels}.
	*/
	power = (int) pow (2.0, Recursion_Levels);
	Elements = (power * power / 2) + 5 * power + 3 * power / 2 - 3;

	Array = (int *) malloc ( sizeof (int) * Elements);

	D_C_1 (Array, Recursion_Levels, 0, Elements - 2, 0, 4);
	Array [Elements - 1] = Elements - 1;

	fd = fopen (filename, "w");
	fprintf (fd, "%d\n", Elements);

	for (i = 0; i < Elements; i++)
		fprintf (fd, "%d\n", Array [i]);

	fclose (fd);

	free (Array);

	return	1;	/* File creation was successful. */
}

/*********************************************************/

void	D_C_1 (int * Array, const int Recursion_Levels, 
			   const int start, const int finish, 
			   const int starting_number, const int starting_hurdle_size)
{
	int		elements, i, power;

	if (Recursion_Levels == 1) {
		/*
		** This is the final step of recursion.
		** Conquer Array and generate Hurdles.
		*/
		elements = finish - start + 1;
		Array [start] = starting_number;
		i = start + 1;
		Array [i] = starting_number + 3;
		Generate_Hurdle (Array, starting_number + 3, i, starting_hurdle_size);
		i += starting_hurdle_size - 1;
		Generate_Hurdle (Array, starting_number + starting_hurdle_size + 2, i, starting_hurdle_size + 1);
		Array [finish - 1] = Array [start] + 2;
		Array [finish] = Array [start] + 1;
	}
	else {
		/*
		** Perform a minor conquest.
		*/
		/* Set Shell. */
		Array [start] = starting_number;
		Array [finish - 1] = starting_number + 2;
		Array [finish] = starting_number + 1;
		/* Set extra element. */
		Array [finish - 2] = starting_number + (finish - start);

		/*
		** Divide Array in 2 sub-regions!
		**
		** Find out how many elements are required in each sub-region.
		*/
		power = (int) pow (2.0, (double) (Recursion_Levels - 2));
		/* Initialize to level-1 shell elemenents and level-(>)1 shell elements. */
		elements = power * 3 + (power - 1) * 4;
		/* #hurdles = 2^{R-1} for each sub-region. */
		power *= 2;
		for (i = starting_hurdle_size; i < starting_hurdle_size + power; i++)
			elements += i;
		elements -= power / 2;

		D_C_1 (Array, Recursion_Levels - 1, start + 1, start + elements, starting_number + 3, starting_hurdle_size);
		D_C_1 (Array, Recursion_Levels - 1, start + elements + 1, finish - 3, starting_number + elements + 3, i);
	}

	return;
}

/*********************************************************/

/*
** Like the previous Divide and Conquer algorithm
** but contains only 7 and 9-element hurdles.
*/

int		Divide_and_Conquer_2 (char * filename, const int Recursion_Levels)
{
	FILE *			fd;
	register int	i;
	register int	power;
	register int	Elements;
	int	*			Array;

	power = (int) pow (2.0, Recursion_Levels);
	Elements = 4 * (power - 1);
	Elements += 16 * power / 2;
	Elements -= power - 1;

	Array = (int *) malloc ( sizeof (int) * Elements);

	D_C_2 (Array, Recursion_Levels, 0, 0);
	Array [Elements - 1] = Elements - 1;

	fd = fopen (filename, "w");
	fprintf (fd, "%d\n", Elements);

	for (i = 0; i < Elements; i++)
		fprintf (fd, "%d\n", Array [i]);

	fclose (fd);

	free (Array);

	return	1;	/* File creation was successful. */
}

/*********************************************************/

void	D_C_2 (int * Array, const int Recursion_Levels, 
			   const int start, const int starting_number)
{
	int		elements, i, power;

	power = (int) pow (2.0, Recursion_Levels);
	elements = 4 * (power - 1);
	elements += 16 * power / 2;
	elements -= power - 1;

	if (Recursion_Levels == 1) {
		/*
		** This is the final step of recursion.
		** Conquer Array and generate Hurdles.
		*/
		Array [start] = starting_number;
		i = start + 1;
		Array [i] = starting_number + 3;
		Generate_Hurdle (Array, starting_number + 3, i, 7);
		i += 6;
		Generate_Hurdle (Array, starting_number + 9, i, 9);
		i += 9;
		Array [i] = Array [start] + 2;
		i++;
		Array [i] = Array [start] + 1;
	}
	else {
		/*
		** Perform a minor conquest.
		*/
		/* Set Shell. */
		Array [start] = starting_number;
		Array [start + elements - 3] = starting_number + 2;
		Array [start + elements - 2] = starting_number + 1;
		/* Set extra element. */
		//Array [start + elements - 4] = starting_number + (elements - 1);

		/*
		** Divide Array in 2 sub-regions!
		**
		** Find out how many elements are required in each sub-region.
		*/
		power = (int) pow (2.0, Recursion_Levels - 1);
		elements = 4 * (power - 1);
		elements += 16 * power / 2;
		elements -= power - 1;

		D_C_2 (Array, Recursion_Levels - 1, start + 1, starting_number + 3);
		D_C_2 (Array, Recursion_Levels - 1, start + elements, starting_number + elements + 2);
		Array [start + 2 * elements - 1] = starting_number + 2 * elements + 1;
	}

	return;
}

/*********************************************************/

/*
** Generate Hurdles and try to avoid monotonically decreasing
** elements within hurdle-bounds.
*/

int		Print_Hurdle (FILE * fileDesc, int starting_number, int elements)
{
	/* Assumes that starting_number has already been printed. */
	/*fprintf (fileDesc, "%d\n", starting_number);*/

	while (elements > 0) {
		if ((elements == 4) || (elements == 5)) {
			/* Monotonically decreasing. */
			if (elements == 5)
				fprintf (fileDesc, "%d\n", starting_number + 3);
			fprintf (fileDesc, "%d\n", starting_number + 2);
			fprintf (fileDesc, "%d\n", starting_number + 1);
			if (elements == 4)
				fprintf (fileDesc, "%d\n", starting_number + 3);
			else
				fprintf (fileDesc, "%d\n", starting_number + 4);
			break;
		}
		else if (elements == 6) {
			/* <1 3 2 5 4 6> */	
			fprintf (fileDesc, "%d\n", starting_number + 2);
			fprintf (fileDesc, "%d\n", starting_number + 1);
			fprintf (fileDesc, "%d\n", starting_number + 4);
			fprintf (fileDesc, "%d\n", starting_number + 3);
			fprintf (fileDesc, "%d\n", starting_number + 5);
			break;
		}
		else if (elements == 7) {
			/* <1 4 3 6 2 5 7> */
			fprintf (fileDesc, "%d\n", starting_number + 3);
			fprintf (fileDesc, "%d\n", starting_number + 2);
			fprintf (fileDesc, "%d\n", starting_number + 5);
			fprintf (fileDesc, "%d\n", starting_number + 1);
			fprintf (fileDesc, "%d\n", starting_number + 4);
			fprintf (fileDesc, "%d\n", starting_number + 6);
			break;
		}
		else if (elements == 8) {
			/* <1 3 6 5 7 4 2 8> */
			fprintf (fileDesc, "%d\n", starting_number + 2);
			fprintf (fileDesc, "%d\n", starting_number + 5);
			fprintf (fileDesc, "%d\n", starting_number + 4);
			fprintf (fileDesc, "%d\n", starting_number + 6);
			fprintf (fileDesc, "%d\n", starting_number + 3);
			fprintf (fileDesc, "%d\n", starting_number + 1);
			fprintf (fileDesc, "%d\n", starting_number + 7);
			break;
		}
		else if (elements == 9) {
			/* <1 6 3 8 5 7 2 4 9> */
			fprintf (fileDesc, "%d\n", starting_number + 5);
			fprintf (fileDesc, "%d\n", starting_number + 2);
			fprintf (fileDesc, "%d\n", starting_number + 7);
			fprintf (fileDesc, "%d\n", starting_number + 4);
			fprintf (fileDesc, "%d\n", starting_number + 6);
			fprintf (fileDesc, "%d\n", starting_number + 1);
			fprintf (fileDesc, "%d\n", starting_number + 3);
			fprintf (fileDesc, "%d\n", starting_number + 8);
			break;
		}
		else if (elements == 10) {
			/* <1 5 9 4 8 3 7 2 6 10> */
			fprintf (fileDesc, "%d\n", starting_number + 4);
			fprintf (fileDesc, "%d\n", starting_number + 8);
			fprintf (fileDesc, "%d\n", starting_number + 3);
			fprintf (fileDesc, "%d\n", starting_number + 7);
			fprintf (fileDesc, "%d\n", starting_number + 2);
			fprintf (fileDesc, "%d\n", starting_number + 6);
			fprintf (fileDesc, "%d\n", starting_number + 1);
			fprintf (fileDesc, "%d\n", starting_number + 5);
			fprintf (fileDesc, "%d\n", starting_number + 9);
			break;
		}
		else {
			/* elements >= 11. */
			/* e.g.: <1 7 2 5 {8} 4 6 3 {10 9 11}> */
			fprintf (fileDesc, "%d\n", starting_number + 6);
			fprintf (fileDesc, "%d\n", starting_number + 1);
			fprintf (fileDesc, "%d\n", starting_number + 4);
			fprintf (fileDesc, "%d\n", starting_number + 7);
			fprintf (fileDesc, "%d\n", starting_number + 3);
			fprintf (fileDesc, "%d\n", starting_number + 5);
			fprintf (fileDesc, "%d\n", starting_number + 2);
			elements -= 7;
			starting_number += 7;
		}
	}

	return	1;	/* File creation was successful. */
}

/*********************************************************/

/*
** Generate Hurdles and try to avoid monotonically decreasing
** elements within hurdle-bounds.
*/

int		Generate_Hurdle (int * Array, const int starting_number, const int starting_position, const int elements)
{
	int		i;

	/* Assumes that starting_number has already been set into Array. */
	/* Array [starting_position] = starting_number; */

	i = starting_position + 1;
	if ((elements == 4) || (elements == 5)) {
		/* Monotonically decreasing. */
		if (elements == 5)
			Array [i++] = starting_number + 3;
		Array [i++] = starting_number + 2;
		Array [i++] = starting_number + 1;
		if (elements == 4)
			Array [i++] = starting_number + 3;
		else
			Array [i++] = starting_number + 4;
	}
	else if (elements == 6) {
		/* <1 3 2 5 4 6> */
		Array [i++] = starting_number + 2;
		Array [i++] = starting_number + 1;
		Array [i++] = starting_number + 4;
		Array [i++] = starting_number + 3;
		Array [i++] = starting_number + 5;
	}
	else if (elements == 7) {
		/* <1 4 3 6 2 5 7> */
		Array [i++] = starting_number + 3;
		Array [i++] = starting_number + 2;
		Array [i++] = starting_number + 5;
		Array [i++] = starting_number + 1;
		Array [i++] = starting_number + 4;
		Array [i++] = starting_number + 6;
	}
	else if (elements == 8) {
		/* <1 3 6 5 7 4 2 8> */
		Array [i++] = starting_number + 2;
		Array [i++] = starting_number + 5;
		Array [i++] = starting_number + 4;
		Array [i++] = starting_number + 6;
		Array [i++] = starting_number + 3;
		Array [i++] = starting_number + 1;
		Array [i++] = starting_number + 7;
	}
	else if (elements == 9) {
		/* <1 6 3 8 5 7 2 4 9> */
		Array [i++] = starting_number + 5;
		Array [i++] = starting_number + 2;
		Array [i++] = starting_number + 7;
		Array [i++] = starting_number + 4;
		Array [i++] = starting_number + 6;
		Array [i++] = starting_number + 1;
		Array [i++] = starting_number + 3;
		Array [i++] = starting_number + 8;
	}
	else if (elements == 10) {
		/* <1 5 9 4 8 3 7 2 6 10> */
		Array [i++] = starting_number + 4;
		Array [i++] = starting_number + 8;
		Array [i++] = starting_number + 3;
		Array [i++] = starting_number + 7;
		Array [i++] = starting_number + 2;
		Array [i++] = starting_number + 6;
		Array [i++] = starting_number + 1;
		Array [i++] = starting_number + 5;
		Array [i++] = starting_number + 9;
	}
	else {
		/* elements >= 11. */
		/* e.g.: <1 7 2 5 {8} 4 6 3 {10 9 11}> */
		Array [i++] = starting_number + 6;
		Array [i++] = starting_number + 1;
		Array [i++] = starting_number + 4;
		Array [i++] = starting_number + 7;
		Array [i++] = starting_number + 3;
		Array [i++] = starting_number + 5;
		Array [i++] = starting_number + 2;
		Generate_Hurdle (Array, starting_number + 7, starting_position + 7, elements - 7);
	}

	return	1;	/* File creation was successful. */
}

/*********************************************************/
